<?php
namespace WhatsBoost;

if (!defined('ABSPATH')) { exit; }

require_once __DIR__ . '/class-admin.php';
require_once __DIR__ . '/class-public.php';
require_once __DIR__ . '/rest/class-rest-controller.php';
require_once __DIR__ . '/woocommerse.php';

final class initBoost {
    public function init(): void {
        // i18n
        $this->load_textdomain();
        // Fallback de traducciones en ausencia de archivos .mo
        add_filter('gettext', [$this, 'gettext_fallback'], 10, 3);
        // Admin is always available to allow introducir la clave
        (new Admin())->hooks();

        $license_key = trim((string) get_option('whatsboost_license_key', ''));
        if ($license_key === '') {
            // Si no hay clave, no cargamos funcionalidades públicas ni rutas REST
            return;
        }

        // Front (no-WooCommerce)
        (new PublicFacing())->hooks();

        // WooCommerce integration
        if (function_exists('wc_get_order')) {
            \WhatsBoost\Woocommerse\init();
        }

        // REST
        add_action('rest_api_init', function () {
            (new Rest\Rest_Controller())->register_routes();
        });
    }

    public function load_textdomain(): void {
        // Intentar compilar automáticamente el .mo desde el .po si no existe.
        $rel = dirname(plugin_basename(WHATSBOOST_FILE)) . '/languages';
        $lang_dir = WP_PLUGIN_DIR . '/' . $rel;

        // Determinar locale que usará WordPress
        $locale = function_exists('determine_locale') ? determine_locale() : get_locale();
        $locale = str_replace('-', '_', (string) $locale);

        $mo_file = $lang_dir . '/whatsboost-' . $locale . '.mo';
        $po_file = $lang_dir . '/whatsboost-' . $locale . '.po';

        if (!file_exists($mo_file) && file_exists($po_file) && is_readable($po_file)) {
            // Intentar compilar el .po a .mo en runtime
            try {
                $this->compile_po_to_mo($po_file, $mo_file);
            } catch (\Exception $e) {
                // No hacemos nada si falla la generación; WordPress usará fallback si existe
            }
        }

        load_plugin_textdomain('whatsboost', false, $rel);
    }

    /**
     * Compila un archivo .po simple a .mo (suficiente para la mayoría de PO planos usados aquí).
     */
    private function compile_po_to_mo(string $poFile, string $moFile): void {
        if (!is_readable($poFile)) { throw new \RuntimeException('PO file not readable'); }
        $contents = file_get_contents($poFile);
        $lines = preg_split('/\r?\n/', $contents);

        $entries = [];
        $state = null;
        $cur = ['msgid' => '', 'msgstr' => ''];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '' || strpos($line, '#') === 0) {
                if ($cur['msgid'] !== '' || $cur['msgstr'] !== '') {
                    $entries[] = $cur;
                    $cur = ['msgid' => '', 'msgstr' => ''];
                    $state = null;
                }
                continue;
            }
            if (preg_match('/^msgid\s+"(.*)"$/', $line, $m)) { $state = 'msgid'; $cur['msgid'] = stripcslashes($m[1]); continue; }
            if (preg_match('/^msgstr\s+"(.*)"$/', $line, $m)) { $state = 'msgstr'; $cur['msgstr'] = stripcslashes($m[1]); continue; }
            if (preg_match('/^"(.*)"$/', $line, $m)) { if ($state && isset($cur[$state])) { $cur[$state] .= stripcslashes($m[1]); } continue; }
        }
        if ($cur['msgid'] !== '' || $cur['msgstr'] !== '') { $entries[] = $cur; }

        $trans = [];
        foreach ($entries as $e) {
            $id = $e['msgid'];
            $str = $e['msgstr'];
            if ($id === '') { continue; }
            $trans[$id] = $str;
        }
        if (empty($trans)) { throw new \RuntimeException('No translations found'); }

        ksort($trans);
        $ids = array_keys($trans);
        $strings = array_values($trans);

        $idsBuf = '';
        $stringsBuf = '';
        foreach ($ids as $id) { $idsBuf .= $id . "\0"; }
        foreach ($strings as $s) { $stringsBuf .= $s . "\0"; }

        $originals = [];
        $translations = [];

        $curIdOffset = 0;
        foreach ($ids as $id) { $len = strlen($id); $originals[] = [$len, $curIdOffset]; $curIdOffset += $len + 1; }
        $curStrOffset = 0;
        foreach ($strings as $s) { $len = strlen($s); $translations[] = [$len, $curStrOffset]; $curStrOffset += $len + 1; }

        $idsLen = strlen($idsBuf);
        $stringsLen = strlen($stringsBuf);

        $n = count($originals);
        $origTableOffset = 28;
        $transTableOffset = $origTableOffset + ($n * 8);
        $idsOffset = $transTableOffset + ($n * 8);
        $stringsOffset = $idsOffset + $idsLen;

        $f = function($v){ return pack('V', $v); };
        $magic = 0x950412de;

        $of = fopen($moFile, 'wb');
        if (!$of) { throw new \RuntimeException('Cannot open MO for writing'); }

        fwrite($of, $f($magic));
        fwrite($of, $f(0)); // revision
        fwrite($of, $f($n));
        fwrite($of, $f($origTableOffset));
        fwrite($of, $f($transTableOffset));
        fwrite($of, $f(0)); // hashtab size

        $cur = $idsOffset;
        for ($i = 0; $i < $n; $i++) {
            fwrite($of, $f($originals[$i][0]));
            fwrite($of, $f($cur));
            $cur += $originals[$i][0] + 1;
        }

        $cur = $stringsOffset;
        for ($i = 0; $i < $n; $i++) {
            fwrite($of, $f($translations[$i][0]));
            fwrite($of, $f($cur));
            $cur += $translations[$i][0] + 1;
        }

        fwrite($of, $idsBuf);
        fwrite($of, $stringsBuf);
        fclose($of);
    }

    /**
     * Fallback de traducciones basado en un mapa interno cuando no existen archivos .mo
     */
    public function gettext_fallback($translated, $text, $domain) {
        if ($domain !== 'whatsboost') { return $translated; }

        $locale = function_exists('determine_locale') ? determine_locale() : get_locale();
        // Normalizar posibles separadores de locale y convertir a string
        $locale = str_replace('-', '_', (string) $locale);
        // Antes: solo aplicábamos fallback para locales en inglés. Ahora soportamos:
        // - Si el locale empieza por 'en' → mapear desde español a inglés (comportamiento previo)
        // - Si el locale empieza por 'es' → mapear desde inglés a español (nuevo)

        // Mapa ES => EN para las cadenas más visibles del plugin (construido en runtime para evitar duplicados en literal)
        static $map = null;
        if ($map === null) {
            $map = [
            // General
            'WhatsBoost' => 'WhatsBoost',
            'Documentación' => 'Documentation',
            'Ajustes de WhatsBoost' => 'WhatsBoost Settings',
            'Configuración' => 'Setup',
            'Configuración completa' => 'Setup complete',
            'Estados de pedido' => 'Order statuses',
            'Configuración de la licencia' => 'License settings',
            // Footer links
            'Nosotros' => 'About us',
            'Política de privacidad' => 'Privacy policy',
            'Términos del servicio' => 'Terms of service',
            'Enlaces de pie de página' => 'Footer links',
            'ALMC logo' => 'ALMC logo',
            // Header quick actions buttons (admin header partial)
            'Ir a WhatsBoost' => 'Go to WhatsBoost',
            'Iniciar sesión' => 'Sign in',
            'Registrarse' => 'Sign up',
            'Acciones rápidas de WhatsBoost' => 'WhatsBoost quick actions',
            'Abrir sitio de WhatsBoost' => 'Open WhatsBoost site',
            'Inicia sesión en WhatsBoost' => 'Sign in to WhatsBoost',
            'Crear una cuenta en WhatsBoost' => 'Create a WhatsBoost account',

                // Settings page / UI
                'Verificar clave' => 'Verify key',
                'Pega aquí tu clave de WhatsBoost.' => 'Paste your WhatsBoost key here.',
                'Pega tu clave para activar el plugin.' => 'Paste your key to activate the plugin.',
                'Clave de licencia' => 'License key',
                'Clave verificada' => 'Key verified',
                'Clave verificada.' => 'Key verified.',
                'Clave de licencia no válida. Inténtalo de nuevo.' => 'Invalid license key. Please try again.',
                'Clave no válida. Intenta con otra clave o vuelve a copiarla.' => 'Invalid key. Try another one or copy it again.',
                'No se pudo validar la clave. Inténtalo de nuevo más tarde.' => 'Could not validate the key. Please try again later.',
                'No se pudo conectar al servidor de verificación. Inténtalo más tarde.' => 'Could not connect to the verification server. Please try again later.',
                'No tienes permisos suficientes para realizar esta acción.' => 'You do not have sufficient permissions to perform this action.',

                // Account / docs banner
                'ID de cuenta detectado:' => 'Detected account ID:',
                'No hay ID de cuenta aún. Verifica tu clave para obtenerlo automáticamente.' => 'No account ID yet. Verify your key to fetch it automatically.',
                'Tu clave está verificada, pero falta que pongas tu número de teléfono para que enviemos WhatsApps desde tu propio número.' => 'Your key is verified, but you still need to set your phone number so we can send WhatsApps from your own number.',
                'Tu clave es válida, pero no hemos recibido el identificador de tu cuenta (el número de WhatsApp desde el que se enviarán los mensajes). Conéctalo en WhatsBoost y vuelve a verificar.' => 'Your key is valid, but we haven\'t received your account identifier (the WhatsApp number from which messages will be sent). Connect it in WhatsBoost and verify again.',
                'Clave verificada. Usaremos tu número/cuenta configurado.' => "Key verified. We'll use your configured number/account.",
                'Te falta poner tu número de teléfono para enviar WhatsApps. Copia esta clave y pégala en whatsboost.php (WHATSBOOST_ACCOUNT):' => 'You need to add your phone number to send WhatsApps. Copy this key and paste it in whatsboost.php (WHATSBOOST_ACCOUNT):',
                'La clave pertenece a otro número/cuenta. Actualiza whatsboost.php con esta clave para enviar desde tu número:' => 'The key belongs to another number/account. Update whatsboost.php with this key to send from your number:',
                // Añadido: traducción para el aviso combinado que aparece en algunos sitios
                'WhatsBoost: Necesitas poner tu número de teléfono para que se envíen los WhatsApps desde tu teléfono.' => 'WhatsBoost: You need to add your phone number so WhatsApps can be sent from your phone.',
                // Subtítulo bajo el selector de número
                'Este es el número que usaremos para enviar WhatsApps a tus clientes. Si lo cambias aquí, los mensajes saldrán de ese número.' => 'This is the number we\'ll use to send WhatsApps to your customers. If you change it here, messages will be sent from that number.',

                // Docs / steps
                '¿Aún no tienes tu número conectado?' => 'Don\'t have your number connected yet?',
                'Sigue estos pasos para conectar tu WhatsApp y que podamos enviar mensajes desde tu número.' => 'Follow these steps to connect your WhatsApp so we can send messages from your number.',
                'Inicia sesión en tu cuenta:' => 'Sign in to your account:',
                'Inicia sesión en WhatsBoost' => 'Sign in to WhatsBoost',
                'Inicio sessión de whatsboost' => 'WhatsBoost sign-in',
                'En el menú de la izquierda, entra en <strong>Hosts (1) → WhatsApp</strong>, o ve directo a' => 'In the left menu, go to <strong>Hosts (1) → WhatsApp</strong>, or go directly to',
                'esta URL' => 'this URL',
                'Luego pulsa <strong>“Añadir cuenta” (2)</strong> como en la captura.' => 'Then click <strong>"Add account" (2)</strong> as in the screenshot.',
                'Paso: Hosts (1) y Añadir cuenta (2)' => 'Step: Hosts (1) and Add account (2)',
                'Haz clic en <strong>Vincular cuenta</strong> y escanea el QR con tu WhatsApp.' => 'Click <strong>Link account</strong> and scan the QR with your WhatsApp.',
                'QR para vincular' => 'QR to link',
                'Paso: Vincular cuenta' => 'Step: Link account',
                'Cuando tu cuenta esté conectada, vuelve aquí y pulsa “Verificar clave”. Detectaremos tu ID automáticamente.' => 'When your account is connected, come back here and click "Verify key". We will detect your ID automatically.',
                'Vídeo: cómo escanear el QR y vincular tu WhatsApp' => 'Video: how to scan the QR and link your WhatsApp',
                'Si no sabes cómo escanear el QR, este vídeo te lo muestra paso a paso.' => 'If you don\'t know how to scan the QR, this video shows it step by step.',
                'Consejo' => 'Tip',
                'Si ya tienes la sesión de WhatsApp Web abierta en tu móvil, el QR se vincula en segundos. Asegúrate de que tu línea aparece como <strong>connected</strong> en el panel.' => 'If you already have an active WhatsApp Web session on your phone, the QR links in seconds. Make sure your line appears as <strong>connected</strong> in the panel.',
                'Tras vincular, el ID único se guardará aquí como “ID de cuenta detectado”.' => 'After linking, the unique ID will be saved here as "Detected account ID".',
                'Cerrar (Esc)' => 'Close (Esc)',

                // Welcome / status page strings
                'Tu clave ha sido verificada correctamente. ¡Ya puedes usar el plugin WhatsBoost!' => 'Your key has been verified successfully. You can now use the WhatsBoost plugin!',
                'Bienvenido WhatsBoost' => 'Welcome to WhatsBoost',
                'Cómo funciona esta página: aquí puedes decidir en qué estados de pedido quieres enviar mensajes por WhatsApp.' => 'How this page works: here you can decide in which order statuses you want to send WhatsApp messages.',
                'Marca (activa) los estados de pedido donde quieres enviar el mensaje. Por ejemplo, “Procesando” y “Completado”.' => 'Check (enable) the order statuses where you want to send the message. For example, "Processing" and "Completed".',
                'Escribe el mensaje en la caja de texto de cada estado. Puedes usar etiquetas como {order_number} o {customer_name}.' => 'Write the message in the text box for each status. You can use tags like {order_number} or {customer_name}.',
                'Pulsa en Guardar cambios al final para que todo se quede guardado.' => 'Click Save Changes at the end to store everything.',
                'Cuando un pedido cambie a ese estado, el plugin preparará el mensaje con tus etiquetas y lo enviará automáticamente.' => 'When an order changes to that status, the plugin will prepare the message with your tags and send it automatically.',
                'Consejo: a la derecha tienes un listado de etiquetas con un botón de copiar. Pégalas en el texto para personalizar tus mensajes.' => 'Tip: on the right you have a list of tags with a copy button. Paste them into the text to personalize your messages.',
                'WooCommerce no está activo. Instálalo y actívalo para ver los estados de pedido.' => 'WooCommerce is not active. Install and activate it to see the order statuses.',

                // Welcome-page help sidebar and examples (new translations)
                'Guardado correctamente.' => 'Saved successfully.',
                'Escribe aquí el texto para este estado…' => 'Write the text for this status here…',
                'Ayuda rápida: etiquetas y botón de copiar' => 'Quick help: tags and copy button',
                'Abajo tienes todas las etiquetas explicadas. Para copiar una, pulsa el botón pequeño con el icono de copiar y luego pégala en el texto de la izquierda. El plugin la cambia por el dato real al enviar.' => 'Below you have all the tags explained. To copy one, click the small button with the copy icon and then paste it into the text on the left. The plugin replaces it with the real data when sending.',
                'Nombre del cliente.' => 'Customer name.',
                'Apellido del cliente.' => 'Customer last name.',
                'Nombre del primer producto del pedido.' => 'Name of the first product in the order.',
                'Cantidad del primer producto.' => 'Quantity of the first product.',
                'Total del pedido con moneda.' => 'Order total with currency.',
                'Método de envío o plazo.' => 'Shipping method or delivery time.',
                'Nombre de tu tienda o marca.' => 'Name of your store or brand.',
                'Enlace para ver el pedido.' => 'Link to view the order.',
                'Número del pedido (p. ej., #123).' => 'Order number (e.g., #123).',
                'Enlace o código de seguimiento (si existe).' => 'Tracking link or code (if available).',
                'Copiar %s' => 'Copy %s',
                'Reglas rápidas' => 'Quick rules',
                'No uses corchetes [ ]. Usa llaves { }.' => 'Do not use square brackets [ ]. Use curly braces { }.',
                'No dejes llaves vacías. Si no tienes el dato, borra esa parte.' => 'Do not leave empty braces. If you do not have the data, remove that part.',
                'debe abrir correctamente.' => 'must open correctly.',
                'Ejemplo mínimo (lo que escribes)' => 'Minimal example (what you write)',
                'Hola {nombre} 👋 Tu {producto_pedido} (x{cantidad}) está en proceso. Mira aquí: {link}' => 'Hi {nombre} 👋 Your {producto_pedido} (x{cantidad}) is being processed. See here: {link}',
                'Resultado de ese ejemplo (lo que verá el cliente)' => 'Result of that example (what the customer will see)',
                'Hola Laura 👋 Tu guantes térmicos (x2) está en proceso. Mira aquí: https://mi-tienda.com/pedido/123' => 'Hi Laura 👋 Your thermal gloves (x2) are being processed. See here: https://my-store.com/order/123',

                // Admin.js generic messages
                'Ha ocurrido un error.' => 'An error occurred.',
                'Operación realizada.' => 'Operation completed.',
                'Error de red.' => 'Network error.',
                'Verificando...' => 'Verifying...',
                'Descartar' => 'Dismiss',

                // License notice and docs
                'Para empezar a enviar WhatsApps necesitas introducir tu clave de licencia de WhatsBoost y asociar tu número. Es rápido y seguro — la obtienes en pocos pasos desde la documentación o la sección de ajustes.' => 'To start sending WhatsApps you need to enter your WhatsBoost license key and link your number. It\'s quick and safe — you get it in a few steps from the documentation or the settings section.',
                'Abrir ajustes de WhatsBoost' => 'Open WhatsBoost settings',
                'Cómo conseguir la clave' => 'How to get the key',
                'No se encontró la vista de documentación.' => 'Documentation view not found.',
                'Tu clave ha sido verificada. ¡Todo listo!' => 'Your key has been verified. All set!',

                // Privacy policy
                'realiza solicitudes externas solo cuando pegas y verificas tu clave o cuando esta ya ha sido verificada, para comprobar el estado de la cuenta asociada.' => 'makes external requests only when you paste and verify your key or when it has already been verified, to check the status of the associated account.',
                'Se envía tu clave (secreta) al servicio de WhatsBoost para validar la suscripción y recuperar el identificador de la cuenta (unique).' => 'Your (secret) key is sent to the WhatsBoost service to validate the subscription and retrieve the account identifier (unique).',
                'El plugin almacena en la base de datos de WordPress: la clave, una marca de verificación, el identificador de cuenta y un transient de control de comprobaciones.' => 'The plugin stores in the WordPress database: the key, a verification flag, the account identifier and a transient to control checks.',
                'En la desinstalación, el plugin elimina todas estas opciones y transients.' => 'On uninstall, the plugin deletes all these options and transients.',
                'No se recopilan otros datos personales desde WordPress ni se envían datos del cliente/pedidos a menos que configures y utilices el servicio de WhatsBoost para enviar mensajes, en cuyo caso se envían únicamente los datos necesarios para el mensaje.' => 'No other personal data is collected from WordPress nor are customer/order data sent unless you configure and use the WhatsBoost service to send messages, in which case only the data necessary for the message are sent.',

                // Requested mapping
                'Todo correcto. Enviaremos WhatsApps desde tu número.' => 'All good. We will send WhatsApps from your number.',
                "Vas a reemplazar la clave guardada por la nueva.\n\nSi continúas, la clave anterior se sustituirá por esta.\n\n¿Quieres continuar?" => "You are about to replace the saved key with the new one.\n\nIf you continue, the previous key will be replaced by this one.\n\nDo you want to proceed?",

                // Number picker and accounts table (ES -> EN)
                'Número para enviar WhatsApps' => 'Number to send WhatsApps from',
                'Solo verás en este desplegable las cuentas que funcionan (conectadas). Por defecto usaremos la primera que funcione; si tienes varias, elige aquí la que quieras usar.' => 'You will only see accounts that work (connected) in this dropdown. By default we will use the first one that works; if you have several, choose here the one you want to use.',
                'Cuentas de WhatsApp' => 'WhatsApp accounts',
                'Teléfono' => 'Phone',
                'ID de cuenta' => 'Account ID',
                'Estado' => 'Status',
                'Funciona' => 'Working',
                'No funciona' => 'Not working',
                'Si un número aparece como <strong>No funciona</strong>, es porque no está conectado en <a href="https://whatsboost.net/dashboard/hosts/whatsapp" target="_blank" rel="noopener">WhatsBoost → Hosts → WhatsApp</a>. Entra allí y vuelve a vincularlo; cuando esté conectado, aparecerá aquí como <strong>Funciona</strong>.' => 'If a number appears as <strong>Not working</strong>, it is because it is not connected in <a href="https://whatsboost.net/dashboard/hosts/whatsapp" target="_blank" rel="noopener">WhatsBoost → Hosts → WhatsApp</a>. Go there and link it again; when it is connected, it will appear here as <strong>Working</strong>.',
                'Actualizar lista' => 'Refresh list',
                'No hay cuentas conectadas. Solo aparecen aquí las que funcionan.' => 'There are no connected accounts. Only those that are working appear here.',
                'Cargando cuentas...' => 'Loading accounts...'
            ];
        }

        // Construir mapa inverso (EN => ES) automáticamente a partir del mapa ES=>EN
        static $rev = null;
        if ($rev === null) {
            $rev = [];
            foreach ($map as $es => $en) {
                if (!isset($rev[$en])) { // preferimos la primera correspondencia
                    $rev[$en] = $es;
                }
            }
        }

        // Si el locale es español, intentar mapear cadenas en INGLÉS -> ESPAÑOL
        if (stripos($locale, 'es') === 0) {
            return $rev[$text] ?? $translated;
        }

        // Si el locale es inglés, mantener el comportamiento previo (ES -> EN)
        if (stripos($locale, 'en') === 0) {
            return $map[$text] ?? $translated;
        }

        // Otros locales: devolver lo que WordPress ya tenga (no interferimos)
        return $translated;
    }
}
